
/**
 * EMemory.h - (c) EliteMMO Network, 2015
 *
 * This unpublished material is proprietary to EliteMMO Network.
 * All rights reserved.
 *
 * The methods and techniques described herein are considered trade secrets
 * and/or condifential. Reproduction or distribution, in whole or in part,
 * is forbidden except by express written permission of EliteMMO Network.
 *
 * This file is part of the EliteMMO Development Kit.
 */

#ifndef __EDK_THREADING_H_INCLUDED__
#define __EDK_THREADING_H_INCLUDED__

#if defined (_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#include <Windows.h>

namespace EliteMMO
{
    namespace Threading
    {
        enum ThreadPriorty
        {
            Lowest = -2,
            BelowNormal = -1,
            Normal = 0,
            AboveNormal = 1,
            Highest = 2
        };

        class ELockableObject
        {
            CRITICAL_SECTION m_CriticalSection;
            ELockableObject(const ELockableObject&) = delete;

        public:
            ELockableObject(void)
            {
                ::InitializeCriticalSection(&this->m_CriticalSection);
            }
            virtual ~ELockableObject(void)
            {
                ::DeleteCriticalSection(&this->m_CriticalSection);
            }

        public:
            void Lock(void)     { ::EnterCriticalSection(&this->m_CriticalSection); }
            void Unlock(void)   { ::LeaveCriticalSection(&this->m_CriticalSection); }
        };

        class EEvent
        {
            HANDLE m_EventHandle;
            EEvent(const EEvent&) = delete;

        public:
            EEvent(bool bManualReset = false)
            {
                this->m_EventHandle = ::CreateEvent(nullptr, bManualReset, FALSE, nullptr);
            }
            virtual ~EEvent(void)
            {
                if (this->m_EventHandle != nullptr)
                    ::CloseHandle(this->m_EventHandle);
                this->m_EventHandle = nullptr;
            }

        public:
            bool Reset(void) { return (::ResetEvent(this->m_EventHandle) ? true : false); }
            bool Raise(void) { return (::SetEvent(this->m_EventHandle) ? true : false); }
            bool IsSignaled(void) { return (::WaitForSingleObject(this->m_EventHandle, 0) == WAIT_OBJECT_0); }
            bool WaitForEvent(DWORD dwMilliseconds) { return (::WaitForSingleObject(this->m_EventHandle, dwMilliseconds) == WAIT_OBJECT_0); }
        };

        class EThread
        {
            HANDLE      m_ThreadHandle;
            DWORD       m_ThreadId;
            EEvent      m_StartEvent;
            EEvent      m_EndEvent;
            EThread(const EThread&) = delete;

        public:
            EThread(void)
                : m_ThreadHandle(nullptr), m_ThreadId(0), m_StartEvent(true), m_EndEvent(true)
            { }
            virtual ~EThread(void)
            {
                if (this->m_ThreadHandle != nullptr)
                    this->Stop();
            }

        public:
            virtual DWORD EThreadEntry(void) = 0;

        public:
            DWORD InternalThreadEntry(void)
            {
                if (this->IsTerminated())
                    return 0;

                this->m_EndEvent.Reset();
                ::Sleep(10);
                this->m_StartEvent.Raise();

                return this->EThreadEntry();
            }

            void Start(void)
            {
                this->m_StartEvent.Reset();
                this->m_ThreadHandle = ::CreateThread(nullptr, 0, ThreadCallback, (LPVOID)this, 0, &this->m_ThreadId);
            }

            void Stop(void)
            {
                this->RaiseEnd();
                if (this->WaitForFinish(INFINITE))
                {
                    ::CloseHandle(this->m_ThreadHandle);
                    this->m_ThreadHandle = nullptr;
                    this->m_ThreadId = 0;
                }
            }

            bool WaitForFinish(DWORD dwMilliseconds = INFINITE)
            {
                if (this->m_ThreadHandle == nullptr)
                    return false;
                return (::WaitForSingleObject(this->m_ThreadHandle, dwMilliseconds) != WAIT_TIMEOUT);
            }

            void SetPriority(ThreadPriorty priority)
            {
                if (this->m_ThreadHandle != nullptr)
                    ::SetThreadPriority(this->m_ThreadHandle, priority);
            }

            int GetPriority(void)
            {
                if (this->m_ThreadHandle == nullptr)
                    return ThreadPriorty::Normal;
                return ::GetThreadPriority(this->m_ThreadHandle);
            }

            void RaiseEnd(void) { this->m_EndEvent.Raise(); }
            void ResetEnd(void) { this->m_EndEvent.Reset(); }
            bool IsTerminated(void) { return this->m_EndEvent.IsSignaled(); }

        public:
            HANDLE GetHandle(void) const { return this->m_ThreadHandle; }
            DWORD GetId(void) const { return this->GetId(); }
            
            DWORD GetExitCode(void) const 
            {
                if (this->m_ThreadHandle == nullptr)
                    return 0;

                DWORD dwExitCode = 0;
                ::GetExitCodeThread(this->m_ThreadHandle, &dwExitCode);
                return dwExitCode;
            }

        private:
            static DWORD __stdcall ThreadCallback(LPVOID lpParam)
            {
                auto thread = (EThread*)lpParam;
                return thread->InternalThreadEntry();
            }
        };
    }; // namespace Threading
}; // namespace EliteMMO

#endif // __EDK_THREADING_H_INCLUDED__
